<?php
// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set headers to prevent caching and ensure JSON response
header('Cache-Control: no-cache, must-revalidate');
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Include the database connection file
require_once 'db_connect.php';

// ----------------------
// 1. GET FILTER PARAMETERS
// ----------------------
// Get all filter values from the URL parameters
// Convert numeric values to appropriate types
$min_price = isset($_GET['min_price']) && $_GET['min_price'] !== '' ? (float)$_GET['min_price'] : null;
$max_price = isset($_GET['max_price']) && $_GET['max_price'] !== '' ? (float)$_GET['max_price'] : null;
$min_grams = isset($_GET['min_grams']) && $_GET['min_grams'] !== '' ? (int)$_GET['min_grams'] : null;
$max_grams = isset($_GET['max_grams']) && $_GET['max_grams'] !== '' ? (int)$_GET['max_grams'] : null;
$shine = isset($_GET['shine']) && $_GET['shine'] !== '' ? $_GET['shine'] : null;
$base = isset($_GET['base']) && $_GET['base'] !== '' ? $_GET['base'] : null;

// Log the processed parameters
error_log("Processed parameters: " . print_r([
    'min_price' => $min_price,
    'max_price' => $max_price,
    'min_grams' => $min_grams,
    'max_grams' => $max_grams,
    'shine' => $shine,
    'base' => $base
], true));

// ----------------------
// 2. BUILD SQL QUERY
// ----------------------
// Start with a base query that joins paint_types with main_colors
$sql = "SELECT p.id, p.name, p.price, p.grams, p.shine_level, p.base_type, 
               p.photo, p.main_color_id, mc.name AS main_color_name 
        FROM paint_types p 
        JOIN main_colors mc ON mc.id = p.main_color_id 
        WHERE 1=1";
$params = [];

// Add conditions for each filter if it's set
if ($min_price !== null) {
    $sql .= " AND p.price >= ?";
    $params[] = $min_price;
}
if ($max_price !== null) {
    $sql .= " AND p.price <= ?";
    $params[] = $max_price;
}
if ($min_grams !== null) {
    $sql .= " AND p.grams >= ?";
    $params[] = $min_grams;
}
if ($max_grams !== null) {
    $sql .= " AND p.grams <= ?";
    $params[] = $max_grams;
}
if ($shine !== null) {
    $sql .= " AND p.shine_level = ?";
    $params[] = $shine;
}
if ($base !== null) {
    $sql .= " AND p.base_type = ?";
    $params[] = $base;
}

// Add ORDER BY to ensure consistent results
$sql .= " ORDER BY p.id";

// Log the final SQL query and parameters
error_log("SQL Query: " . $sql);
error_log("Query parameters: " . print_r($params, true));

// ----------------------
// 3. EXECUTE QUERY
// ----------------------
try {
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $filtered_products = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Log the number of results
    error_log("Number of products found: " . count($filtered_products));
} catch (PDOException $e) {
    error_log("Database error: " . $e->getMessage());
    echo json_encode(['error' => 'Database error occurred: ' . $e->getMessage()]);
    exit;
}

// ----------------------
// 4. GET RAL COLORS
// ----------------------
// For each product, get its associated RAL colors
// We use &$product to modify the array by reference
foreach ($filtered_products as &$product) {
    // Query to get all RAL colors for this product's main color
    $ralSql = "SELECT rc.* FROM ral_colors rc 
               WHERE rc.main_color_id = ?";
    $ralStmt = $pdo->prepare($ralSql);
    $ralStmt->execute([$product['main_color_id']]);
    // Add the RAL colors to the product data
    $product['rals'] = $ralStmt->fetchAll(PDO::FETCH_ASSOC);
}

// ----------------------
// 5. RETURN RESULTS
// ----------------------
echo json_encode($filtered_products);
exit;
?>